;----------------------------------------------------------------------------
;GetSwitch: Fills an ASCIIZ buffer with the command line parameter value.
;On entry:
;  (SP+0)=Ret address
;  (SP+2)=Address of cmdline ASCIIZ string, preceded by its len in ONE byte
;  (SP+4)=Address of ASCIIZ parameter to find its value
;  (SP+6)=Address of ASCIIZ buffer to fill with the parameter's value
;On return:
;  Carry flag reset=OK
;  Carry flag set=Error (no chars in cmdline, etc)

GetSwitch:	POP	HL		;Fetch return address in HL
		POP	DE		;Fetch cmdline address in DE
		POP	BC		;Fetch param address in BC
		EX	(SP),HL		;Push ret addr, buffer addr to HL
		LD	(GS_Param),BC	;Save param ASCIIZ address
		LD	(GS_OutBf),HL	;Save output buffer address
		EX	DE,HL		;CmdLine to HL, OutBuff to DE
		LD	C,(HL)		;CmdLine len to C (just-a-byte)
		LD	B,0		;Hi-byte of BC is zero
		INC	HL		;Point to first char in CmdLine
GS_GetSlash:	LD	A,"/"		;Char to find to A
		CPIR			;Find the slash
		JR	Z,GS_ChkParam	;Slash found. Check param.
		JR	GS_Error	;Outta chars or no slashes!
GS_ChkParam:	LD	DE,(GS_Param)	;Param name address to DE
		DEC	HL		;Compensate for first INC HL
GS_CP_Loop:	DEC	BC		;Take char into account
		INC	HL		;Point to next char
		LD	A,(DE)		;Param name char to A
		INC	DE		;Point to next char in param name
		OR	A		;Is End Of String?
		JR	Z,GS_Found	;Jump if so: Param found!
		CP	(HL)		;Same as (HL)?
		JR	Z,GS_CP_Loop	;Continue if so
		XOR	020h		;Toggle uppercase/lowercase bit
		CP	(HL)		;Same as (HL)?
		JR	Z,GS_CP_Loop	;Jump if so
		JR	GS_GetSlash	;Bad param: Continue search for "/"
GS_Found:	LD	A,(HL)		;Char should be "="
		CP	"="		;Is it?
		JR	NZ,GS_GetSlash	;Jump back if not: Wrong param
		INC	HL		;Point to char after equals sign
		LD	DE,(GS_OutBf)	;DE points to start of buffer
GS_CopyBuff:	LD	A,(HL)		;Move value char to A
		CP	" "		;Is space?
		JR	Z,GS_Done	;Jump if so. Transfer finished
		CP	00h		;End of cmdline?
		JR	Z,GS_Done	;Jump if so. No more to transfer
		LD	(DE),A		;Copy char to output buffer
		INC	HL		;Point to next char
		INC	DE		;Point to next placeholder
		JR	GS_CopyBuff	;Loop back until done
GS_Done:	XOR	A		;Reset A
		LD	(DE),A		;Terminate the string  
		RET			;Done :)
GS_Error:	SCF			;Flag "ERROR!!!"
		RET			;Done :(

GS_OutBf:	DB	00h,00h		;Output buffer address
GS_Param:	DB	00h,00h		;Parameter name address
		
;----------------------------------------------------------------------------
;ExpandMacros: Expands "%1" to "%9" type macros in an ASCII buffer with the
;nth ASCIIZ string from another buffer, leaving the ASCII (not ASCIIZ) in a
;third supplied address.
;On entry:
;  (SP+0)=Ret address
;  (SP+2)=Pointer to expanded buffer start (will be ASCII, not ASCIIZ)
;  (SP+4)=Pointer to a series of ASCIIZ strings (parameters, from 1 to 9)
;  (SP+6)=Pointer to the source, "to expand" ASCIIZ buffer
;On return:

ExpandMacros:	POP	HL		;Fetch ret addr briefly
		POP	DE		;Fetch OutBuffer in DE
		POP	BC		;Fetch ParBuffer in BC
		EX	(SP),HL		;Exchg Ret to (SP) and RawBuff to HL
EM_SavePtr:	LD	(EM_PBuff),BC	;Save param buffer ptr
EM_Loop:	LD	A,(HL)		;Char to A
		INC	HL		;Point to next char
		OR	A		;Is it a zero?
		RET	Z		;Finished if so
		CP	"%"		;Is it the macro indicator?
		JR	NZ,EM_CopyChr	;Jump if not
		LD	A,(HL)		;Char after "%" to A
		CP	"0"		;Is char > "0"?
		JR	NC,EM_ChkNine	;Jump if so
EM_CopyPct:	LD	A,"%"		;Restore percentage sign in A
		JR	EM_CopyChr	;And transfer it to out buffer
EM_ChkNine:	CP	"9"		;Is char lower than "9"?
		JR	NC,EM_CopyPct	;Jump if not to transfer the "%"
		SUB	"0"		;Now is 0 <= A <= 9
		PUSH	HL		;Save input buffer pointer
		LD	HL,(EM_PBuff)	;Get Param Buffer address in HL
		LD	B,A		;Param number to B
		XOR	A		;Reset A to zero
EM_FindPar:	PUSH	BC		;Save counter in B
		LD	BC,0		;Prepare to loop thru 64K
		CPIR			;Look for a zero char
		POP	BC		;Restore zeroes counter
		DJNZ	EM_FindPar	;Loop back until param is found
EM_CopyPar:	LD	A,(HL)		;Param char to A
		INC	HL		;Point to next char in param
		OR	A		;Is it zero?
		JR	NZ,EM_CopyPar2	;Go on to copy char
		POP	HL		;Restore main in buffer pointer
		INC	HL		;Skip param number
		JR	EM_Loop		;And continue looping
EM_CopyPar2:	LD	(DE),A		;Copy char in param to output buffer
		INC	DE		;Advance out buffer pointer
		JR	EM_CopyPar	;And loop thru all chars in param
EM_CopyChr:	LD	(DE),A		;Copy char in main to output buffer
		INC	DE		;Advance out buffer pointer
		JR	EM_Loop		;And loop thru all chars in main

EM_PBuff:	DW	0000h		;Param buffer save zone

;----------------------------------------------------------------------------
;Ascii2Word: Converts an ASCIIZ buffer with numbers to a word.
;On entry:
;  (SP+0)=Ret address
;  (SP+2)=ASCIIZ buffer address
;On return:
;  Carry flag reset: Operation was successful.
;   A=Lo-byte of converted word
;  HL=Converted word
;  Carry flag set: Error (overflow or invalid char in buffer)
;Carry flag will be set in case of overflow or error. In case of overflow,
;the contents of A and HL on return are undefined.
;The ASCIIZ buffer may contain a base-number prefix: "#" indicates HEX
;buffer numbers, "@" means OCTAL, "%" means binary. Default is DECIMAL.

Ascii2Word:	POP	HL		;Fetch ret addr briefly
		EX	(SP),HL		;Exchg Ret (SP) with BufferAddr (HL)
A2W_Init:	LD	DE,0		;Reset...
		LD	(A2W_RetVal),DE	;...return value
		LD	A,(HL)		;Get first char in buffer
		LD	C,0Fh		;Prepare for HEX
		CP	"#"		;Is HEX prefix?
		JR	Z,A2W_Nxt	;Jump if so
		LD	C,07h		;Prepare for OCT
		CP	"@"		;Is OCT prefix?
		JR	Z,A2W_Nxt	;Jump if so
		LD	C,01h		;Prepare for BIN
		CP	"%"		;Is BIN prefix?
		JR	Z,A2W_Nxt	;Jump if so
		LD	C,09h		;Ok, number is DEC
		DEC	HL		;Compensate for next INC HL
A2W_Nxt:	INC	HL		;Point to next char in buffer
		LD	A,(HL)		;Char to A
		CP	0h		;End of string?
		JR	Z,A2W_Done	;Jump if so
		PUSH	HL		;Save buffer address while adding
		LD	HL,(A2W_RetVal)	;Get return value so far
		PUSH	HL		;Copy HL to...
		POP	DE		;... register DE
		LD	B,C		;Get loop index to B
A2W_Add:	ADC	HL,DE		;Perform sum...
		DJNZ	A2W_Add		;... B times
		JR	C,A2W_Ovfl	;Abort if overflow
		SUB	"0"		;Convert ASCII in A to number
		CP	9		;Less than 9?
		JR	C,A2W_Chk	;Jump if so.
		RES	5,A		;Force char uppercase (just in case)
		SUB	7		;Deduct "A" - "9" diff.
A2W_Chk:	LD	E,A		;Save number to E
		LD	D,0		;Reset hi-byte of DE
		INC	C		;Test if...
		CP	C		;... A > C, and...
		JR	NC,A2W_Ovfl	;... abort if so...
		DEC	C		;... 'cos it's invalid char.
		XOR	A		;Clear carry flag for next ADC
		ADC	HL,DE		;Add the number to the return value
		JR	C,A2W_Ovfl	;Jump if overflow
		LD	(A2W_RetVal),HL	;Store result so far
		POP	HL		;Fetch back string pointer
		JR	A2W_Nxt		;Keep looping
A2W_Ovfl:	SCF			;Set carry flag to say "Error!"
		POP	HL		;Stack balancing
		RET			;Return to caller
A2W_Done:	LD	HL,(A2W_RetVal)	;Fetch return value
		LD	A,L		;Lo-byte of result to A
		RET			;Back to caller

A2W_RetVal:	DEFW	0000h		;Return value storage

;----------------------------------------------------------------------------
;Word2Hex: Prepares a 16 bit number to be printed in HEX form.
;On entry:
;  (SP)=Ret address
;  (SP+2)=Buffer address to fill with ASCII (not ASCIIZ), 4 bytes long
;  (SP+4)=Number
;On return:
;   A=ASCII for lo-nibble of LSB
;  DE=Number
;  HL=Pointer to last char in buffer

Word2Hex:	POP	DE		;Get ret address temporarily
		POP	HL		;Get Buffer address
		LD	(W2HBuf),HL	;Save Buffer address
		POP	HL		;Get Number
		LD	(W2HNum),HL	;Save Number
		PUSH	DE		;Push return address
		LD	L,H		;MSB to LSB
		PUSH	HL		;Push MSB
		LD	HL,(W2HBuf)	;Retrieve buffer pointer
		PUSH	HL		;Push buffer address
		CALL	Byte2Hex	;Fill half buffer
		INC	HL		;HL points to third char in buffer
		LD	DE,(W2HNum)	;DE now holds LSB of Number
		PUSH	DE		;Number to Stack
		PUSH	HL		;Buffer to Stack
		CALL	Byte2Hex	;Fill rest of buffer
		RET			;Done.

W2HBuf:		DW	0000h
W2HNum:		DW	0000h

;----------------------------------------------------------------------------
;Byte2Hex: Prepares an 8 bit number to be printed in HEX form.
;On entry:
;  (SP)=Ret address
;  (SP+2)=Buffer address to fill with ASCII (not ASCIIZ), 2 bytes long
;  (SP+4)=LSB=Number, MSB=ignored
;On return:
;   A=ASCII for lo-nibble
;   D=MSB of Number (ignored)
;   E=Number
;  HL=Pointer to last char in buffer

Byte2Hex:	POP	HL		;Get ret address temporarily
		POP	DE		;Get Buffer address
		LD	(B2HBuf),DE	;Save Buffer address
		POP	DE		;Get Number
		PUSH	HL		;Push return value
		LD	HL,(B2HBuf)	;Make HL point to buffer
		LD	A,E		;Number to A (H=0, and ignored)
		RRCA			;Get hi-nibble of A into lo-nibble
		RRCA
		RRCA
		RRCA
		AND	0Fh		;Isolate lo-nibble
		ADD	A,"0"		;Add ASCII for zero
		CP	3Ah		;Greater than "9"?
		JR	C,B2HBig1	;Jump if so
		ADD	A,07h		;Add ASCII difference to "A"
B2HBig1:	LD	(HL),A		;Save char to buffer
		INC	HL		;Point to next char in buffer
		LD	A,E		;Rretrieve number
		AND	0Fh		;Isolate lo-nibble
		ADD	A,"0"		;Add ASCII for "0"
		CP	3Ah		;Greater than "9"?
		JR	C,B2HBig2	;Jump if so
		ADD	A,07h		;Add ASCII difference to "A"
B2HBig2:	LD	(HL),A		;Save char to buffer
		RET			;Done.

B2HBuf:		DW	0000h

;----------------------------------------------------------------------------
;Byte2Ascii: Prepares a 8 bit number to be printed in decimal form.
;On entry:
;  (SP+0)=Ret address
;  (SP+2)=ASCII (not ASCIIZ) buffer address, 3 bytes long.
;  (SP+4)=Number to convert (in LSB, MSB is ignored)
;On return:
;   A=ASCII of least significant digit
;   C=ASCII of mid significant digit (or space, if it was ZERO)
;  HL=Points to last char in buffer

Byte2Ascii:	POP	BC		;Fetch ret address temporarily
		POP	HL		;Fetch ASCII buffer address
		POP	DE		;Fetch nmber to convert in E
		PUSH	BC		;Push back ret address
		LD	A,E		;Get number to convert in A
		LD	C,"0"-1		;Prepare character
B2A_3:		INC	C		;Inc char
		SUB	100		;Subtract 100 from A
		JR	NC,B2A_3	;Keep looping until negative
		ADD	A,100		;Restore positive value
		LD	E,A		;Save value temporarily
		LD	A,C		;Hundreds char to A
		CP	"0"		;Is zero?
		JR	NZ,B2A_Hundreds	;Jump if not
		LD	C," "		;Preset to put a space if Hundreds=0
B2A_Hundreds:	LD	(HL),C		;Save hundreds char
		LD	A,E		;Retrieve old value
		INC	HL		;Point to next char
		LD	C,"0"-1		;Prepare character
B2A_2:		INC	C		;Inc char
		SUB	10		;Subtract 10 from A
		JR	NC,B2A_2	;Keep looping until negative
		ADD	A,"0"+10	;Restore positive value (and ASCII)
		LD	E,A		;Save char briefly
		LD	A,C		;Tens char to A
		CP	"0"		;Is zero?
                JR	NZ,B2A_Tens	;Jump if not
		LD	C," "		;Change char to be a space
		LD	A,E		;Retrieve last char
B2A_Tens:	LD	(HL),C		;Save tens char
		INC	HL		;Point to next char
		LD	(HL),A		;Save units char (don't check for 0)
		RET			;Done

;----------------------------------------------------------------------------
;Word2Ascii: Prepares a 16 bit number to be printed in decimal form.
;On entry:
;  (SP+0)=Ret address
;  (SP+2)=Buffer to fill with ASCII (not ASCIIZ), 5 bytes long
;  (SP+4)=Number to convert to ASCII
;On return:
;   A=ASCII of last digit printed
;  BC=000Ah
;  DE=Buffer start address
;  HL=Least significant digit of converted number
;  Any other registers: Not affected

;Word2Ascii:	POP	BC		;Retrieve ret address
;		POP	DE		;Fetch ASCII buffer
;		POP	HL		;Fetch number to convert
;		PUSH	BC		;Push ret address back
Word2Ascii:	POP	HL		;Ret addr briefly to HL
		POP	DE		;Fetch ASCII buffer to DE
		EX	(SP),HL		;Exchg Ret (SP) with Number (HL)
		LD	BC,10000	;Prepare BC to subtract
		PUSH	IX		;Save content of IX
		PUSH	DE		;Transfer content of
		POP	IX		;Register DE to IX
		LD	(IX+0),"0"-1	;Insert char in buffer
		XOR	A		;Clear carry flag
W2A_5:		INC	(IX+0)		;Inc char
                SBC	HL,BC		;Deduct 10K from HL
		JR	NC,W2A_5	;Loop until negative result
		ADD	HL,BC		;Restore positive in HL
		LD	BC,1000		;Prepare BC to subtract
		LD	(IX+1),"0"-1	;Insert char in buffer
		XOR	A		;Clear carry flag
W2A_4:		INC	(IX+1)		;Inc char
		SBC	HL,BC		;Deduct 1K from HL
		JR	NC,W2A_4	;Loop until negative result
		ADD	HL,BC		;Restore positive in HL
		LD	BC,100		;Prepare BC to subtract
		LD	(IX+2),"0"-1	;Insert char in buffer
		XOR	A		;Clear carry flag
W2A_3:		INC	(IX+2)		;Inc char
		SBC	HL,BC		;Deduct 100 from HL
		JR	NC,W2A_3	;Loop until negative result
		ADD	HL,BC		;Restore positive in HL
		LD	BC,10		;Prepare BC to subtract
		LD	(IX+3),"0"-1	;Insert char in buffer
		XOR	A		;Clear carry flag
W2A_2:		INC	(IX+3)		;Inc char
		SBC	HL,BC		;Deduct 10 from HL
		JR	NC,W2A_2	;Loop until negative result
		ADD	HL,BC		;Restore positive to HL
		LD	A,L		;Last number to A
		ADD	A,"0"		;Add the "0" difference
		LD	(IX+4),A	;Put last char in buffer
		POP	IX		;Restore original value in IX
		RET


